import React, { useEffect, useState } from "react";
import { Disclosure } from "@headlessui/react";
import { ChevronDownIcon } from "@heroicons/react/20/solid";
import { apiFetch } from "../utils/api";
import { BOOLEAN_KEYS } from "../constants/booleanKeys";
import { flatToTree, TreeUser } from "../utils/faltTree";

type BoolKey = typeof BOOLEAN_KEYS[number];
type Config = Record<string, any>;
type Draft = Partial<Record<BoolKey, boolean>>;

// Grouped keys for better organization
const GROUPED_KEYS = {
  "Site Features": [
    "sports", "casino", "lotto", "vipclub", "clubmembership", "bonus", "affiliate",
    "crash", "originals", "livegames", "slotsgames", "alllivegames", "allslotsgames",
    "lotterygames", "indiangames", "cards", "instantgames", "esports", "provablyfair"
  ],
  "Cryptocurrencies": [
    "btc", "eth", "ltc", "bch", "usdt", "trx", "doge", "ada", "xrp", "bnb", 
    "usdp", "nexo", "mkr", "tusd", "usdc", "busd", "nc", "npr", "shib", "matic", "sc"
  ],
  "Game Providers": [
    "spribe", "evolution", "pragmaticslots", "pragmaticlive", "ideal",
    "microgaming", "pgsoft", "hacksawgaming", "jili", "netent"
  ]
};

type BoolSwitchProps = {
  label: string;
  checked: boolean;
  onChange: React.ChangeEventHandler<HTMLInputElement>;
};

const BoolSwitch: React.FC<BoolSwitchProps> = ({ label, checked, onChange }) => (
  <div className="flex items-center justify-between py-2">
    <span className="text-gray-700 capitalize">{label}</span>
    <label className="relative inline-flex items-center cursor-pointer">
      <input
        type="checkbox"
        checked={checked}
        onChange={onChange}
        className="sr-only peer"
      />
      <div className="w-11 h-6 bg-gray-200 rounded-full peer peer-checked:bg-blue-600 transition-colors">
        <div className="w-5 h-5 bg-white rounded-full shadow transform transition-transform peer-checked:translate-x-5" />
      </div>
    </label>
  </div>
);

export default function UserConfig() {
  const [tree, setTree] = useState<TreeUser[]>([]);
  const [cfgs, setCfgs] = useState<Record<number, Config>>({});
  const [drafts, setDrafts] = useState<Record<number, Draft>>({});
  const [saving, setSaving] = useState<number | null>(null);

  useEffect(() => {
    apiFetch<TreeUser[]>("/api/admin/users/tree")
      .then(list => setTree(flatToTree(list as any)));
  }, []);

  const loadCfg = async (id: number) => {
    if (cfgs[id]) return;
    setCfgs({ ...cfgs, [id]: await apiFetch(`/api/admin/config/user/${id}`) });
  };

  const toggle = (uid: number, k: BoolKey) => {
    const c = cfgs[uid];
    if (!c) return;
    setCfgs({ ...cfgs, [uid]: { ...c, [k]: !c[k] } });
    setDrafts({ ...drafts, [uid]: { ...(drafts[uid] || {}), [k]: !c[k] } });
  };

  const save = async (uid: number) => {
    const d = drafts[uid];
    if (!d || !Object.keys(d).length) return;
    setSaving(uid);
    try {
      setCfgs({
        ...cfgs,
        [uid]: await apiFetch(`/api/admin/config/user/${uid}`, {
          method: "PUT",
          body: JSON.stringify(d)
        })
      });
      const { [uid]: _, ...rest } = drafts;
      setDrafts(rest);
    } finally {
      setSaving(null);
    }
  };

  const render = (u: TreeUser) => (
    <Disclosure key={u.id}>
      {({ open }) => (
        <div className="bg-white rounded-lg shadow-md mb-4">
          <Disclosure.Button
            className="flex w-full items-center justify-between p-4 bg-gray-50 hover:bg-gray-100 transition-colors rounded-t-lg"
            onClick={() => loadCfg(u.id)}
          >
            <div className="flex items-center gap-3">
              <span className="font-semibold text-gray-800">{u.name}</span>
              <span className="text-sm text-gray-500">{u.email}</span>
            </div>
            <ChevronDownIcon className={`h-5 w-5 text-gray-600 transition-transform ${open ? "rotate-180" : ""}`} />
          </Disclosure.Button>

          <Disclosure.Panel className="p-6 space-y-6">
            {cfgs[u.id] ? (
              <>
                {Object.entries(GROUPED_KEYS).map(([group, keys]) => (
                  <div key={group} className="bg-gray-50 rounded-lg p-4">
                    <h2 className="text-lg font-semibold text-gray-800 mb-3">{group}</h2>
                    <div className="grid gap-2 sm:grid-cols-2 lg:grid-cols-3">
                      {keys.map(k => (
                        <BoolSwitch
                          key={k}
                          label={k}
                          checked={!!cfgs[u.id][k]}
                          onChange={() => toggle(u.id, k as BoolKey)}
                        />
                      ))}
                    </div>
                  </div>
                ))}
                <button
                  disabled={!drafts[u.id] || saving === u.id}
                  onClick={() => save(u.id)}
                  className="px-6 py-2 bg-blue-600 text-white rounded-lg font-semibold 
                    hover:bg-blue-700 transition-colors disabled:opacity-50 
                    disabled:cursor-not-allowed flex items-center space-x-2"
                >
                  {saving === u.id && (
                    <svg className="animate-spin h-5 w-5 text-white" viewBox="0 0 24 24">
                      <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" />
                      <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z" />
                    </svg>
                  )}
                  <span>{saving === u.id ? "Saving..." : "Save Changes"}</span>
                </button>
              </>
            ) : (
              <div className="text-gray-600 text-lg animate-pulse">Loading...</div>
            )}
          </Disclosure.Panel>

          {u.children.length > 0 && (
            <div className="ml-6">
              {u.children.map(render)}
            </div>
          )}
        </div>
      )}
    </Disclosure>
  );

  return (
     <div className="ml-0 lg:ml-[256px] p-2">
    <div className="max-w-7xl mx-auto p-6 space-y-8 bg-gray-50 min-h-screen">
      <h1 className="text-3xl font-bold text-gray-900">User Site Configuration</h1>
      {tree.length === 0 ? (
        <p className="text-gray-600 text-lg">No users found.</p>
      ) : (
        tree.map(render)
      )}
    </div>
    </div>
  );
}