import React, { useState, useEffect } from 'react';
import axios from 'axios';

// Define types
interface KycApplication {
  id: number;
  userId: string;
  firstName: string;
  lastName: string;
  gender: string;
  dob: string;
  address: string;
  city: string;
  country: string;
  documentType: string;
  idFrontPath?: string;
  idBackPath?: string;
  passportPath?: string;
  status: 'Pending' | 'Verified' | 'Rejected';
  rejectionReason?: string;
  createdAt: string;
}

const Kyc: React.FC = () => {
  const [applications, setApplications] = useState<KycApplication[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [selectedApp, setSelectedApp] = useState<KycApplication | null>(null);
  const [rejectionReason, setRejectionReason] = useState<string>('');
  const [filter, setFilter] = useState<'All' | 'Pending' | 'Verified' | 'Rejected'>('Pending');

  const apiBaseUrl = 'https://api.bitcoinjito.com/kyc';

  // Fetch all KYC applications
  useEffect(() => {
    const fetchApplications = async () => {
      setLoading(true);
      try {
        // This endpoint would need to be added to your API
        const response = await axios.get(`${apiBaseUrl}/admin/applications`);
        setApplications(response.data);
      } catch (error) {
        console.error('Failed to fetch KYC applications:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchApplications();
    
    // Refresh data every 2 minutes
    const interval = setInterval(fetchApplications, 120000);
    return () => clearInterval(interval);
  }, []);

  // Filter applications based on status
  const filteredApplications = applications.filter(app => 
    filter === 'All' ? true : app.status === filter
  );

  // Update KYC status - Fixed to ensure type safety
  const updateStatus = async (userId: string, status: 'Verified' | 'Rejected') => {
    try {
      await axios.put(`${apiBaseUrl}/update-status`, {
        userId,
        status,
        rejectionReason: status === 'Rejected' ? rejectionReason : undefined
      });
      
      // Update local state with fixed type safety
      setApplications(prevApps => 
        prevApps.map(app => {
          if (app.userId === userId) {
            return {
              ...app,
              status,
              rejectionReason: status === 'Rejected' ? rejectionReason : undefined
            };
          }
          return app;
        })
      );
      
      // Clear selected application and rejection reason
      setSelectedApp(null);
      setRejectionReason('');
      
    } catch (error) {
      console.error('Failed to update KYC status:', error);
    }
  };
  const getFilenameFromPath = (path: string | undefined): string | undefined => {
    if (!path) return undefined;
    // Extract just the filename from the path
    const parts = path.split('/');
    return parts[parts.length - 1];
  };

  // Get document URL for display - Fixed return type to string | undefined
  const getDocumentUrl = (path: string | undefined): string | undefined => {
    if (!path) return undefined;
    
    // Extract the filename from the path
    const filename = getFilenameFromPath(path);
    if (!filename) return undefined;
    
    // Add file extension if missing
    const filenameWithExt = filename.includes('.') ? filename : `${filename}.png`;
    
    // Log the URL being generated
    const url = `${apiBaseUrl}/documents/${filenameWithExt}`;
    console.log('Generated document URL:', url);
    
    return url;
  };

  return (
    <div className='ml-0 lg:ml-[220px] p-5 '>
       <div className="flex flex-col min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-blue-600 text-white shadow-lg">
        <div className="container mx-auto py-4 px-6">
          <h1 className="text-2xl font-bold">KYC Verification Admin Panel</h1>
        </div>
      </header>

      {/* Main Content */}
      <div className="container mx-auto flex flex-col md:flex-row flex-1 p-6 gap-6">
        {/* Left Side - Applications List */}
        <div className="w-full md:w-1/3 bg-white rounded-lg shadow-md overflow-hidden">
          <div className="p-4 border-b border-gray-200">
            <h2 className="text-lg font-semibold">Applications</h2>
            <div className="mt-2 flex gap-2">
              {(['All', 'Pending', 'Verified', 'Rejected'] as const).map((status) => (
                <button
                  key={status}
                  className={`px-3 py-1 rounded-full text-sm ${
                    filter === status
                      ? 'bg-blue-600 text-white'
                      : 'bg-gray-200 text-gray-700'
                  }`}
                  onClick={() => setFilter(status)}
                >
                  {status}
                </button>
              ))}
            </div>
          </div>
          
          <div className="overflow-y-auto h-96">
            {loading ? (
              <div className="flex justify-center items-center h-full">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
              </div>
            ) : filteredApplications.length === 0 ? (
              <div className="p-6 text-center text-gray-500">
                No {filter !== 'All' ? filter.toLowerCase() : ''} applications found
              </div>
            ) : (
              <ul className="divide-y divide-gray-200">
                {filteredApplications.map((app) => (
                  <li
                    key={app.id}
                    className={`p-4 cursor-pointer hover:bg-gray-50 ${
                      selectedApp?.id === app.id ? 'bg-blue-50' : ''
                    }`}
                    onClick={() => setSelectedApp(app)}
                  >
                    <div className="flex justify-between items-center">
                      <div>
                        <p className="font-medium">{app.firstName} {app.lastName}</p>
                        <p className="text-sm text-gray-500">ID: {app.userId}</p>
                      </div>
                      <span
                        className={`px-2 py-1 rounded-full text-xs ${
                          app.status === 'Pending'
                            ? 'bg-yellow-100 text-yellow-800'
                            : app.status === 'Verified'
                            ? 'bg-green-100 text-green-800'
                            : 'bg-red-100 text-red-800'
                        }`}
                      >
                        {app.status}
                      </span>
                    </div>
                  </li>
                ))}
              </ul>
            )}
          </div>
        </div>

        {/* Right Side - Application Details */}
        <div className="w-full md:w-2/3 bg-white rounded-lg shadow-md overflow-hidden">
          {selectedApp ? (
            <div className="h-full flex flex-col">
              <div className="p-4 border-b border-gray-200 flex justify-between items-center">
                <h2 className="text-lg font-semibold">
                  Application Details - {selectedApp.firstName} {selectedApp.lastName}
                </h2>
                <span
                  className={`px-3 py-1 rounded-full text-sm ${
                    selectedApp.status === 'Pending'
                      ? 'bg-yellow-100 text-yellow-800'
                      : selectedApp.status === 'Verified'
                      ? 'bg-green-100 text-green-800'
                      : 'bg-red-100 text-red-800'
                  }`}
                >
                  {selectedApp.status}
                </span>
              </div>

              <div className="p-6 flex-1 overflow-y-auto">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  {/* Personal Information */}
                  <div className="space-y-4">
                    <h3 className="text-md font-medium">Personal Information</h3>
                    <div className="bg-gray-50 p-4 rounded-lg space-y-2">
                      <div className="grid grid-cols-2">
                        <span className="text-gray-500">Full Name</span>
                        <span>{selectedApp.firstName} {selectedApp.lastName}</span>
                      </div>
                      <div className="grid grid-cols-2">
                        <span className="text-gray-500">Gender</span>
                        <span>{selectedApp.gender}</span>
                      </div>
                      <div className="grid grid-cols-2">
                        <span className="text-gray-500">Date of Birth</span>
                        <span>{new Date(selectedApp.dob).toLocaleDateString()}</span>
                      </div>
                      <div className="grid grid-cols-2">
                        <span className="text-gray-500">Address</span>
                        <span>{selectedApp.address}</span>
                      </div>
                      <div className="grid grid-cols-2">
                        <span className="text-gray-500">City</span>
                        <span>{selectedApp.city}</span>
                      </div>
                      <div className="grid grid-cols-2">
                        <span className="text-gray-500">Country</span>
                        <span>{selectedApp.country}</span>
                      </div>
                      <div className="grid grid-cols-2">
                        <span className="text-gray-500">Document Type</span>
                        <span className="capitalize">{selectedApp.documentType}</span>
                      </div>
                    </div>
                  </div>

                  {/* Document Verification */}
                  <div className="space-y-4">
                    <h3 className="text-md font-medium">Document Verification</h3>
                    <div className="bg-gray-50 p-4 rounded-lg space-y-4">
                      {selectedApp.documentType === 'idcard' ? (
                        <>
                          <div>
                            <p className="text-gray-500 mb-2">ID Card Front</p>
                            {selectedApp.idFrontPath ? (
                              <img
                                src={getDocumentUrl(selectedApp.idFrontPath)}
                                alt="ID Card Front"
                                className="w-full h-48 object-contain border rounded"
                              />
                            ) : (
                              <div className="border rounded p-4 text-center text-gray-500">
                                No front ID image found
                              </div>
                            )}
                          </div>
                          <div>
                            <p className="text-gray-500 mb-2">ID Card Back</p>
                            {selectedApp.idBackPath ? (
                              <img
                                src={getDocumentUrl(selectedApp.idBackPath)}
                                alt="ID Card Back"
                                className="w-full h-48 object-contain border rounded"
                              />
                            ) : (
                              <div className="border rounded p-4 text-center text-gray-500">
                                No back ID image found
                              </div>
                            )}
                          </div>
                        </>
                      ) : (
                        <div>
                          <p className="text-gray-500 mb-2">Passport</p>
                          {selectedApp.passportPath ? (
                            <img
                              src={getDocumentUrl(selectedApp.passportPath)}
                              alt="Passport"
                              className="w-full h-48 object-contain border rounded"
                            />
                          ) : (
                            <div className="border rounded p-4 text-center text-gray-500">
                              No passport image found
                            </div>
                          )}
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                {/* Rejection Reason (if applicable) */}
                {selectedApp.status === 'Rejected' && selectedApp.rejectionReason && (
                  <div className="mt-6">
                    <h3 className="text-md font-medium text-red-600">Rejection Reason</h3>
                    <div className="bg-red-50 p-4 rounded-lg mt-2 text-red-700">
                      {selectedApp.rejectionReason}
                    </div>
                  </div>
                )}
              </div>

              {/* Action Buttons */}
              {selectedApp.status === 'Pending' && (
                <div className="p-4 border-t border-gray-200 bg-gray-50">
                  <div className="flex flex-col space-y-4">
                    <div>
                      <label htmlFor="rejection-reason" className="block text-sm font-medium text-gray-700">
                        Rejection Reason (required if rejecting)
                      </label>
                      <textarea
                        id="rejection-reason"
                        rows={3}
                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        value={rejectionReason}
                        onChange={(e) => setRejectionReason(e.target.value)}
                        placeholder="Explain why this application is being rejected..."
                      />
                    </div>
                    <div className="flex justify-end space-x-3">
                      <button
                        className="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 disabled:bg-red-300"
                        onClick={() => updateStatus(selectedApp.userId, 'Rejected')}
                        disabled={!rejectionReason.trim()}
                      >
                        Reject
                      </button>
                      <button
                        className="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700"
                        onClick={() => updateStatus(selectedApp.userId, 'Verified')}
                      >
                        Approve
                      </button>
                    </div>
                  </div>
                </div>
              )}
            </div>
          ) : (
            <div className="h-full flex justify-center items-center text-gray-500 p-8">
              <div className="text-center">
                <svg
                  className="mx-auto h-12 w-12 text-gray-400"
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"
                  />
                </svg>
                <h3 className="mt-2 text-lg font-medium">No application selected</h3>
                <p className="mt-1">Select an application from the list to view details</p>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
      </div>
   
  );
};

export default Kyc;