import React, { useState, useEffect } from 'react';
import axios from 'axios';
import { Bar, Pie } from 'react-chartjs-2';
import { 
  Chart as ChartJS, 
  CategoryScale, 
  LinearScale, 
  BarElement, 
  Title, 
  Tooltip, 
  Legend,
  ArcElement
} from 'chart.js';

ChartJS.register(
  CategoryScale,
  LinearScale,
  BarElement,
  Title,
  Tooltip,
  Legend,
  ArcElement
);

type User = {
  id: string;                // normalized to string
  name: string;
  email: string;
  country: string;
  phone: string;             // normalized to string
  total_deposit_usd: number; // normalized from *_usdt if needed
  total_withdrawal_usd: number;
};

const Users: React.FC = () => {
  const [users, setUsers] = useState<User[]>([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [snackbar, setSnackbar] = useState<string | null>(null);

  const [depositWithdrawalData, setDepositWithdrawalData] = useState({
    labels: [] as string[],
    datasets: [
      {
        label: 'Total Deposits',
        data: [] as number[],
        backgroundColor: 'rgba(79, 70, 229, 0.6)',
        borderColor: 'rgba(79, 70, 229, 1)',
        borderWidth: 1,
      },
      {
        label: 'Total Withdrawals',
        data: [] as number[],
        backgroundColor: 'rgba(156, 163, 175, 0.6)',
        borderColor: 'rgba(156, 163, 175, 1)',
        borderWidth: 1,
      },
    ],
  });

  const [userActivityData, setUserActivityData] = useState({
    labels: ['Active Users', 'Inactive Users'],
    datasets: [
      {
        data: [0, 0],
        backgroundColor: [
          'rgba(79, 70, 229, 0.8)',
          'rgba(156, 163, 175, 0.8)',
        ],
        borderColor: [
          'rgba(79, 70, 229, 1)',
          'rgba(156, 163, 175, 1)',
        ],
        borderWidth: 1,
      },
    ],
  });

  useEffect(() => {
    fetchUsersData();
  }, []);

  const normalizeUser = (u: any): User => {
    const deposit =
      Number(u.total_deposit_usd ?? u.total_deposit_usdt ?? 0) || 0;
    const withdrawal =
      Number(u.total_withdrawal_usd ?? u.total_withdrawal_usdt ?? 0) || 0;

    return {
      id: String(u.id ?? ''),
      name: String(u.name ?? '—'),
      email: String(u.email ?? '—'),
      country: String(u.country ?? ''),
      phone: String(u.phone ?? ''),
      total_deposit_usd: deposit,
      total_withdrawal_usd: withdrawal,
    };
  };

  const fetchUsersData = async () => {
    try {
      const response = await axios.get('https://api.bitcoinjito.com/getUserData');
      const data = response.data;

      if (Array.isArray(data)) {
        const normalized = data.map(normalizeUser).reverse();
        setUsers(normalized);

        // Charts: use normalized data and USD fields
        const top = normalized.slice(0, 7);
        const labels = top.map(u => u.name || u.email || u.id);
        const deposits = top.map(u => u.total_deposit_usd);
        const withdrawals = top.map(u => u.total_withdrawal_usd);

        setDepositWithdrawalData({
          labels,
          datasets: [
            {
              label: 'Total Deposits',
              data: deposits,
              backgroundColor: 'rgba(79, 70, 229, 0.6)',
              borderColor: 'rgba(79, 70, 229, 1)',
              borderWidth: 1,
            },
            {
              label: 'Total Withdrawals',
              data: withdrawals,
              backgroundColor: 'rgba(156, 163, 175, 0.6)',
              borderColor: 'rgba(156, 163, 175, 1)',
              borderWidth: 1,
            },
          ],
        });

        const activeUsers = normalized.filter(u => u.total_deposit_usd > 0).length;
        const inactiveUsers = normalized.length - activeUsers;
        setUserActivityData({
          labels: ['Active Users', 'Inactive Users'],
          datasets: [
            {
              data: [activeUsers, inactiveUsers],
              backgroundColor: [
                'rgba(79, 70, 229, 0.8)',
                'rgba(156, 163, 175, 0.8)',
              ],
              borderColor: [
                'rgba(79, 70, 229, 1)',
                'rgba(156, 163, 175, 1)',
              ],
              borderWidth: 1,
            },
          ],
        });
      } else {
        console.error('Unexpected response structure:', data);
        setUsers([]);
      }
    } catch (error) {
      console.error('Failed to fetch user data:', error);
      setUsers([]);
    }
  };

  const handleDeleteUser = async (id: string, name: string) => {
    try {
      await axios.delete('https://api.bitcoinjito.com/deleteUser', { data: { id } });
      await fetchUsersData();
      setSnackbar(`User ${name} deleted successfully`);
      setTimeout(() => setSnackbar(null), 3000);
    } catch (error) {
      console.error('Failed to delete user:', error);
    }
  };

  const q = searchTerm.trim().toLowerCase();
  const filteredUsers = q
    ? users.filter(u => {
        const haystack = [
          u.id || '',
          u.name || '',
          u.email || '',
          u.phone || '',
        ]
          .map(s => s.toLowerCase())
          .join(' ');
        return haystack.includes(q);
      })
    : users;

  const currency = (n: number) => `$${(Number(n) || 0).toLocaleString(undefined, { maximumFractionDigits: 2 })}`;

  return (
    <div className='ml-0 lg:ml-[256px] p-8 '>
      {snackbar && (
        <div className="mb-4 text-center text-white bg-green-500 py-2 px-4 rounded-lg shadow-md">
          {snackbar}
        </div>
      )}

      <h1 className="text-2xl text-white font-semibold mb-4">{`Total Users: ${users.length}`}</h1>

      <div className="mb-5 flex justify-end">
        <label htmlFor="search" className="sr-only">Search</label>
        <div className="relative w-1/2">
          <input
            type="text"
            id="search"
            className="bg-[#37384A] w-full pl-10 pr-4 py-2 rounded-lg border border-gray-200 shadow-sm focus:outline-none focus:ring-2 focus:border-transparent text-white placeholder-gray-400"
            placeholder="Search by email, name or UID"
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
          />
          <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
            <svg className="h-5 w-5 text-gray-500" fill="none" strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" viewBox="0 0 24 24" stroke="currentColor">
              <path d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
            </svg>
          </div>
        </div>
      </div>

      <div className="overflow-x-auto relative shadow-md sm:rounded-lg bg-white">
        <table className="w-full text-sm text-left text-gray-500">
          <thead className="text-xs text-white uppercase bg-[#646ECD]">
            <tr>
              <th className="py-3 px-6">S.No</th>
              <th className="py-3 px-6">Name</th>
              <th className="py-3 px-6">Email</th>
              <th className="py-3 px-6">UID</th>
              <th className="py-3 px-6">Phone</th>
              <th className="py-3 px-6">Total Deposit (USDT)</th>
              <th className="py-3 px-6">Total Withdrawal (USDT)</th>
              <th className="py-3 px-6">PNL (USDT)</th>
              <th className="py-3 px-6">Action</th>
            </tr>
          </thead>
          <tbody>
            {filteredUsers.map((user, index) => {
              const pnl = (user.total_deposit_usd || 0) - (user.total_withdrawal_usd || 0);
              const pnlClass = pnl >= 0 ? 'text-green-600' : 'text-red-600';
              return (
                <tr key={`${user.id}-${index}`} className="group bg-[#37384A] border-b border-gray-200 hover:bg-gray-50">
                  <td className="text-white group-hover:text-gray-600 py-4 px-6">{index + 1}</td>
                  <td className="text-white group-hover:text-gray-600 py-4 px-6">{user.name}</td>
                  <td className="text-white group-hover:text-gray-600 py-4 px-6">{user.email}</td>
                  <td className="text-white group-hover:text-gray-600 py-4 px-6">{user.id}</td>
                  <td className="text-white group-hover:text-gray-600 py-4 px-6">{user.phone}</td>
                  <td className="group-hover:text-gray-600 py-4 px-6 text-green-600">{currency(user.total_deposit_usd)}</td>
                  <td className="group-hover:text-gray-600 py-4 px-6 text-red-600">{currency(user.total_withdrawal_usd)}</td>
                  <td className={`group-hover:text-gray-600 py-4 px-6 ${pnlClass}`}>
                    {currency(pnl)}
                  </td>
                  <td className="py-4 px-6">
                    <button
                      onClick={() => handleDeleteUser(user.id, user.name)}
                      className="text-white bg-red-500 hover:bg-red-700 font-medium rounded-lg text-sm px-4 py-2 focus:outline-none focus:ring-2 focus:ring-red-500 focus:ring-offset-2"
                    >
                      Delete
                    </button>
                  </td>
                </tr>
              );
            })}
            {filteredUsers.length === 0 && (
              <tr>
                <td colSpan={9} className="py-6 text-center text-gray-400">No matching users.</td>
              </tr>
            )}
          </tbody>
        </table>
      </div>
    </div>
  );
};

export default Users;
