// src/hooks/useTree.ts
import { useState, useEffect } from "react";
import { apiFetch } from "../utils/api";

/* ────────────────────────────────── Types ───────────────────────────────── */

export interface Node {
  id: number;
  name: string;
  role: { name: string; level: number };
  balance: number;
  parent_id: number | null;
}

export interface Analytics {
  staffTrend:  { day: string; count: number }[];
  playerTrend: { day: string; count: number }[];
}

export interface Metrics {
  staff_cnt:  number;
  player_cnt: number;
  bank:       string;
}

/* ───────────────────────────────── Hook ─────────────────────────────────── */

export function useTree() {
  const [staff,     setStaff]     = useState<Node[]>([]);
  const [players,   setPlayers]   = useState<Node[]>([]);
  const [metrics,   setMetrics]   = useState<Metrics | null>(null);
  const [analytics, setAnalytics] = useState<Analytics | null>(null);
  const [loading,   setLoading]   = useState(true);

  useEffect(() => {
    let cancelled = false;

    (async () => {
      try {
        // optional ?id query-param
        const id   = localStorage.getItem("currentUserId");
        const q    = id && !isNaN(+id) ? `?id=${id}` : "";

        const [s, p, m, a] = await Promise.all([
          apiFetch<Node[]>("/api/staff"),
          apiFetch<Node[]>("/api/staff/players"),
          apiFetch<Metrics>(`/api/staff/metrics/${id}`),
          apiFetch<Analytics>(`/api/staff/analytics/${id}`)
        ]);

        if (cancelled) return;           // component unmounted

        setStaff(s);
        setPlayers(p);
        setMetrics(m);
        setAnalytics(a);
      } catch {
        /* errors (401 etc.) are already handled globally in apiFetch */
      } finally {
        if (!cancelled) setLoading(false);
      }
    })();

    return () => { cancelled = true; };
  }, []);

  return { staff, players, metrics, analytics, loading };
}
