var fetch = require('node-fetch');
var config = require('./config');
var CryptoUtils = require('./cryptoUtils');
var logger = require('./logger');
var db = require('../db');

function GameController() {}

GameController.BASE_URL = config.gameApi.baseUrl;
GameController.API_KEY = config.gameApi.apiKey;
GameController.API_SECRET = config.gameApi.apiSecret;

// Helper to add auth headers
GameController.addAuthHeaders = function(req, body) {
  var timestamp = Date.now().toString();
  var params = Object.assign({}, body, { timestamp: timestamp });
  var signature = CryptoUtils.generateSignature(params, GameController.API_SECRET);
  return {
    'x-api-key': GameController.API_KEY,
    'x-timestamp': timestamp,
    'x-signature': signature,
    'Content-Type': 'application/json'
  };
};

// Launch Game

GameController.getGameLaunchURL = async function(req, res) {
  const { game_uid, _id: uuid, currency_code, language = 'en' } = req.body;

  if (!game_uid || !uuid || !currency_code) {
    return res.status(400).json({ success: false, error: 'Missing required fields' });
  }

  try {
    // 1️⃣ Fetch both id and name in one go
    const userResult = await db.query(
      'SELECT id, name FROM users WHERE uuid = $1',
      [uuid]
    );
    if (userResult.rowCount === 0) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }

    const userId   = userResult.rows[0].id;    // ← numeric bigint
    const userName = userResult.rows[0].name;  // ← the name you wanted

    // 2️⃣ Fetch the SQL balance
    const balRes = await db.query(
      'SELECT * FROM credits WHERE uid = $1',
      [userId]
    );
    if (balRes.rowCount === 0) {
      return res.status(404).json({ success: false, error: 'No credits record found' });
    }

    const credit_amount = (balRes.rows[0][currency_code.toLowerCase()] || 0).toString();

    // 3️⃣ Build your payload
    const payload = {
      game_uid,
      user_id:        userId.toString(),
      credit_amount,
      currency_code,
      language,
      name:           userName
    };

    // 4️⃣ Call the third-party launch endpoint
    const response = await fetch(`${GameController.BASE_URL}launch`, {
      method:  'POST',
      headers: GameController.addAuthHeaders(req, payload),
      body:    JSON.stringify(payload),
    });
    if (!response.ok) {
      throw new Error(`Launch API returned ${response.status}`);
    }

    const data = await response.json();
    if (!data.success || !data.data?.game_launch_url) {
      return res.status(400).json({ success: false, error: 'Game under maintenance.' });
    }

    const { game_launch_url, session_token } = data.data;

    // 5️⃣ Save the session in SQL
    await db.query(
      `INSERT INTO game_sessions
         (user_id, game_uid, session_token, launch_url)
       VALUES ($1,$2,$3,$4)`,
      [userId, game_uid, session_token, game_launch_url]
    );

    logger.info('Game launched', { userId, game_uid });
    return res.json({
      success: true,
      data:    { game_launch_url, session_token }
    });

  } catch (err) {
    logger.error('getGameLaunchURL error', { error: err, body: req.body });
    return res.status(500).json({ success: false, error: 'Server error' });
  }
};
var CurrencyUtils = {
  convertFromUSD: function(usdAmount, currency, callback) {
    if (currency === 'usd') {
      return callback(null, Number(usdAmount.toFixed(8)));
    }

    db.query(
      'SELECT usd_rate FROM exchangerate WHERE currency = $1',
      [currency.toUpperCase()], // Ensure currency matches table (e.g., USDT, INR)
      function(err, result) {
        if (err || !result.rows.length) {
          return callback(new Error('No exchange rate found for ' + currency));
        }

        var usdRate = parseFloat(result.rows[0].usd_rate);
        // Convert USD to target currency: amount / usd_rate
        var amountInCurrency = usdAmount / usdRate;
        callback(null, Number(amountInCurrency.toFixed(8)));
      }
    );
  }
};
// GameController.processBetCallback = function(req, res) {
//   const { user_id, transaction_type, amount, currency, balance, transaction_id } = req.body;

//   if (!user_id || !transaction_type || !currency || balance === undefined) {
//     logger.error('Invalid callback payload', { body: req.body });
//     return res.status(400).json({ success: false, error: 'Invalid payload' });
//   }
//   const newBalance = Number(parseFloat(balance).toFixed(8));
//   const currencyColumn = currency.toLowerCase() === 'inr' ? 'inr' : currency.toLowerCase();

//   pg.query(
//     'INSERT INTO game_transactions (user_id, game_uid, transaction_type, amount, currency, external_transaction_id, additional_data) VALUES ($1, $2, $3, $4, $5, $6, $7) RETURNING id',
//     [
//       user_id,
//       req.body.game_uid || 'unknown',
//       transaction_type,
//       amount,
//       currency.toLowerCase(),
//       transaction_id || 'unknown',
//       JSON.stringify(req.body),
//     ],
//     function(err) {
//       if (err) {
//         logger.error('Database error saving transaction', { error: err });
//         return res.status(500).json({ success: false, error: 'Server error' });
//       }
     
//       pg.query('SELECT usdt, inr FROM credits WHERE uid = $1', [user_id], function(err, balanceResult) {
//         if (err) {
//           logger.error('Database error fetching balance', { error: err });
//           return res.status(500).json({ success: false, error: 'Server error' });
//         }

//         if (balanceResult.rows.length) {
//           pg.query(
//             `UPDATE credits SET ${currencyColumn} = $1 WHERE uid = $2 RETURNING ${currencyColumn}`,
//             [newBalance, user_id],
//             function(err, updateResult) {
//               if (err) {
//                 logger.error('Database error updating balance', { error: err });
//                 return res.status(500).json({ success: false, error: 'Server error' });
//               }

//               logger.info('Bet callback processed', { user_id, transaction_type, currency, newBalance });
//               res.json({
//                 success: true,
//                 new_balance: updateResult.rows[0][currencyColumn].toString(),
//               });
//             }
//           );
//         } 
//       });
//     }
//   );
// };

// GameController.processBetCallback = function(req, res) {
//   const { user_id, transaction_type, amount, currency, balance, transaction_id } = req.body;

//   if (!user_id || !transaction_type || !currency || balance === undefined) {
//     logger.error('Invalid callback payload', { body: req.body });
//     return res.status(400).json({ success: false, error: 'Invalid payload' });
//   }
//   const newBalance = Number(parseFloat(balance).toFixed(8));
//   const currencyColumn = currency.toLowerCase() === 'inr' ? 'inr' : currency.toLowerCase();
//   let rakebackAmount = 0;
//   if (transaction_type === 'bet') {
//     if (currency.toLowerCase() === 'inr') 
//     {
//       db.query(
//         'SELECT usd_rate FROM exchangerate WHERE currency = $1',
//         ['INR'],
//         function(err, rateResult) {
//           if (err) {
//             logger.error('Database error fetching exchange rate', { error: err });
//             return res.status(500).json({ success: false, error: 'Server error' });
//           }
          
//           const inrToUsdRate = rateResult.rows[0].usd_rate;
//           const amountInUsdt = parseFloat(amount) * inrToUsdRate;
//           rakebackAmount = Number((amountInUsdt * 0.002).toFixed(8));
//         }
//       );
//     }
//     else
//     {
//       rakebackAmount = Number((parseFloat(amount) * 0.002).toFixed(8));
//     }
//   }
//   db.query(
//     'INSERT INTO game_transactions (user_id, game_uid, transaction_type, amount, currency, external_transaction_id, additional_data) VALUES ($1, $2, $3, $4, $5, $6, $7) RETURNING id',
//     [
//       user_id,
//       req.body.game_uid || 'unknown',
//       transaction_type,
//       amount,
//       currency.toLowerCase(),
//       transaction_id || 'unknown',
//       JSON.stringify(req.body),
//     ],
//     function(err) {
//       if (err) {
//         logger.error('Database error saving transaction', { error: err });
//         return res.status(500).json({ success: false, error: 'Server error' });
//       }
//       // if (transaction_type === 'bet') {
//       //   db.query(
//       //     'UPDATE users SET rakeamount = rakeamount + $1 WHERE id = $2 RETURNING rakeamount',
//       //     [rakebackAmount, user_id],
//       //     function(err, rakebackResult) {
//       //       if (err) {
//       //         logger.error('Database error updating rakeback', { error: err });
//       //         return res.status(500).json({ success: false, error: 'Server error' });
//       //       }

//       //       logger.info('Rakeback updated', { user_id, rakebackAmount, newRakeback: rakebackResult.rows[0] });
//       //     }
//       //   );
//       // }
//       db.query('SELECT usdt, inr FROM credits WHERE uid = $1', [user_id], function(err, balanceResult) {
//         if (err) {
//           logger.error('Database error fetching balance', { error: err });
//           return res.status(500).json({ success: false, error: 'Server error' });
//         }

//         if (balanceResult.rows.length) {
//           db.query(
//             `UPDATE credits SET ${currencyColumn} = $1 WHERE uid = $2 RETURNING ${currencyColumn}`,
//             [newBalance, user_id],
//             function(err, updateResult) {
//               if (err) {
//                 logger.error('Database error updating balance', { error: err });
//                 return res.status(500).json({ success: false, error: 'Server error' });
//               }

//               logger.info('Bet callback processed', { user_id, transaction_type, currency, newBalance });
//               res.json({
//                 success: true,
//                 new_balance: updateResult.rows[0][currencyColumn].toString(),
//               });
//             }
//           );
//         } 
//       });
//     }
//   );
// };

GameController.processBetCallback = function(req, res) {
  const {
    user_id,
    transaction_type,
    amount,
    currency,
    balance   // new balance from payload
  } = req.body;

  if (!user_id || !transaction_type || !currency || balance === undefined) {
    logger.error('Invalid callback payload', { body: req.body });
    return res.status(400).json({ success: false, error: 'Invalid payload' });
  }

  // normalize
  const newBalance      = Number(parseFloat(balance).toFixed(8));
  const currencyColumn  = currency.toLowerCase() === 'inr' ? 'inr' : currency.toLowerCase();

  function insertAndSync() {
    // a) insert the game_transactions record
    db.query(
      `INSERT INTO game_transactions 
         (user_id, game_uid, transaction_type, amount, currency, external_transaction_id, additional_data)
       VALUES ($1,$2,$3,$4,$5,$6,$7)`,
      [
        user_id,
        req.body.game_uid || 'unknown',
        transaction_type,
        amount,
        currencyColumn,
        req.body.transaction_id || 'unknown',
        JSON.stringify(req.body),
      ],
      function(err) {
        if (err) {
          logger.error('Database error saving transaction', { error: err });
          return res.status(500).json({ success: false, error: 'Server error' });
        }

        // b) fetch old balance
        db.query(
          'SELECT usdt, inr FROM credits WHERE uid = $1',
          [user_id],
          function(err, balRes) {
            if (err) {
              logger.error('Database error fetching balance', { error: err });
              return res.status(500).json({ success: false, error: 'Server error' });
            }

            if (!balRes.rows.length) {
              logger.error('No credits row for user', { user_id });
              return res.status(404).json({ success: false, error: 'No credit record' });
            }

            const oldBalance = Number(balRes.rows[0][currencyColumn] || 0);

            // c) update to the new balance
            db.query(
              `UPDATE credits 
                   SET ${currencyColumn} = $1 
                 WHERE uid = $2
               RETURNING ${currencyColumn}`,
              [newBalance, user_id],
              function(err, updRes) {
                if (err) {
                  logger.error('Database error updating balance', { error: err });
                  return res.status(500).json({ success: false, error: 'Server error' });
                }

                const updatedBalance = Number(updRes.rows[0][currencyColumn]);
                logger.info('Bet callback processed', {
                  user_id,
                  transaction_type,
                  currency:     currencyColumn,
                  oldBalance,
                  updatedBalance
                });

                // d) compute delta for Mongo sync
                const delta      = updatedBalance - oldBalance;
                const syncType   = delta >= 0 ? 'deposit' : 'withdraw';
                const syncAmount = Number(Math.abs(delta).toFixed(8));

                // e) lookup the user’s UUID
                db.query(
                  'SELECT uuid FROM users WHERE id = $1',
                  [user_id],
                  function(err, userRes) {
                    if (err) {
                      logger.error('Database error fetching user UUID', { error: err });
                    } else if (userRes.rows.length) {
                      const uuid = userRes.rows[0].uuid;

                      // f) fire off the Mongo sync call
                      fetch(
                        `https://apisky.codefactory.games/users/balance/${uuid}`,
                        {
                          method: 'PUT',
                          headers: { 'Content-Type': 'application/json' },
                          body: JSON.stringify({
                            type:   syncType,
                            amount: syncAmount
                          })
                        }
                      ).catch(err =>
                        logger.error('Mongo sync failed', { error: err })
                      );
                    } else {
                      logger.error('No SQL user row to get UUID for user_id', { user_id });
                    }
                  }
                );

                // g) finally, respond to the callback
                return res.json({
                  success:     true,
                  new_balance: updatedBalance.toString()
                });
              }
            );
          }
        );
      }
    );
  }

  // No more rakeback branch—just insert & sync
  insertAndSync();
};
GameController.getActiveGames = function(req, res) {
    var vendor = req.query.vendor;
    var page = req.query.page || '1';
    var per_page = req.query.per_page || '20';
  
    fetch(GameController.BASE_URL + 'games?vendor=' + (vendor || '') + '&page=' + page + '&per_page=' + per_page, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to fetch games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };
  
  // Search Games (Simple proxy to admin API)
  GameController.userhistory = function(req, res) {
    var userid= req.query.userid || '';
   
  
    var url = 'https://games.ibitplay.com/api/user/'+ 'user/history?user_id=' + userid;
  
    fetch(url, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to search games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };

  GameController.userhistoryAdmin = function(req, res) {

   
  
    var url = 'https://games.ibitplay.com/api/user/'+ 'historyToClient';
  
    fetch(url, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to search games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };

  GameController.searchGames = function(req, res) {
    var keyword = req.query.keyword || '';
    var vendor = req.query.vendor;
    var page = req.query.page || '1';
    var per_page = req.query.per_page || '20';
  
    var url = GameController.BASE_URL + 'games/search?keyword=' + encodeURIComponent(keyword) +
              (vendor ? '&vendor=' + vendor : '') + '&page=' + page + '&per_page=' + per_page;
  
    fetch(url, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to search games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };
module.exports = GameController;