import React, { useEffect, useMemo, useState } from "react";
import axios from "axios";

// ===== Types =====
interface FanWin {
  id: number;
  userid: number;
  fancyname: string;
  selection: string;
  runsodds: number;
  payout: number;
  eventid: number;
  matchid: number;
  created_at: string;
}

interface PaginationState {
  totalCount: number;
  totalPages: number;
  currentPage: number;
  limit: number;
}

interface FanWinResponse {
  data: FanWin[];
  pagination: PaginationState;
}

const FanWins: React.FC = () => {
  const staffId = localStorage.getItem("currentUserId");
  const token = localStorage.getItem("token");

  const [rows, setRows] = useState<FanWin[]>([]);
  const [searchTerm, setSearchTerm] = useState<string>("");
  const [userId, setUserId] = useState<string>("");
  const [loading, setLoading] = useState<boolean>(false);
  const [pagination, setPagination] = useState<PaginationState>({
    totalCount: 0,
    totalPages: 0,
    currentPage: 1,
    limit: 10,
  });

  const BASE_URL = useMemo(() => {
    return (
      (import.meta as any)?.env?.VITE_API_BASE_URL ||
      process.env.REACT_APP_API_BASE_URL ||
      "https://api.bitcoinjito.com"
    );
  }, []);

  useEffect(() => {
    const t = setTimeout(() => fetchData(), 400);
    return () => clearTimeout(t);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [searchTerm]);

  useEffect(() => {
    fetchData();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [pagination.currentPage]);

  const fetchData = async () => {
    setLoading(true);
    try {
      let url = `${BASE_URL}/sportsbetting/fanwins?page=${pagination.currentPage}&limit=${pagination.limit}`;
      if (searchTerm) url += `&search=${encodeURIComponent(searchTerm)}`;
      if (userId) url += `&userId=${encodeURIComponent(userId)}`;

      const headers: Record<string, string> = {
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
        ...(staffId ? { "x-staff-id": staffId } : {}),
      };

      const { data } = await axios.get<FanWinResponse>(url, { headers });
      setRows(data.data);
      setPagination((p) => ({ ...p, ...data.pagination }));
    } catch (e) {
      console.error("Failed to fetch fan wins:", e);
    } finally {
      setLoading(false);
    }
  };

  const handleSearchChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchTerm(e.target.value);
    setPagination((p) => ({ ...p, currentPage: 1 }));
  };
  const handleUserFilterChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setUserId(e.target.value);
    setPagination((p) => ({ ...p, currentPage: 1 }));
  };
  const handlePageChange = (newPage: number) => {
    if (newPage < 1 || (pagination.totalPages && newPage > pagination.totalPages)) return;
    setPagination((p) => ({ ...p, currentPage: newPage }));
  };
  const handleClearFilters = () => {
    setSearchTerm("");
    setUserId("");
    setPagination((p) => ({ ...p, currentPage: 1 }));
  };

  const formatCurrency = (n: number) =>
    typeof n === "number" && !Number.isNaN(n) ? n.toFixed(2) : "-";
  const formatDate = (iso: string) => new Date(iso).toLocaleString();

  if (loading && rows.length === 0) {
    return (
      <div className="ml-0 lg:ml-[256px] p-6">
        <div className="flex items-center justify-center h-64">
          <div className="text-gray-500">Loading fan wins...</div>
        </div>
      </div>
    );
  }

  return (
    <div className="ml-0 lg:ml-[256px] p-6">
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <h1 className="text-2xl font-bold mb-6">Fan wins</h1>

        {/* Filters */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
          <div>
            <label htmlFor="search-input" className="block text-sm font-medium text-gray-700 mb-1">Search</label>
            <input
              id="search-input"
              type="text"
              placeholder="Fancy, selection, event ID or match ID"
              value={searchTerm}
              onChange={handleSearchChange}
              className="w-full p-2 border rounded focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label htmlFor="user-id-input" className="block text-sm font-medium text-gray-700 mb-1">User ID</label>
            <input
              id="user-id-input"
              type="text"
              placeholder="Filter by User ID"
              value={userId}
              onChange={handleUserFilterChange}
              className="w-full p-2 border rounded focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
        </div>

        {/* Info */}
        <div className="mb-4">
          <span className="text-gray-600">
            Showing {rows.length} of {pagination.totalCount} records
          </span>
          {(searchTerm || userId) && (
            <button
              onClick={handleClearFilters}
              className="ml-4 text-blue-500 hover:text-blue-700"
              aria-label="Clear filters"
            >
              Clear filters
            </button>
          )}
        </div>

        {/* No results */}
        {rows.length === 0 && !loading && (
          <div className="text-center py-10">
            <p className="text-gray-500">No fan wins found</p>
          </div>
        )}

        {/* Table */}
        {rows.length > 0 && (
          <div className="overflow-x-auto">
            <table className="w-full bg-white shadow-md rounded-lg overflow-hidden">
              <thead className="bg-gray-100 border-b">
                <tr>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">User ID</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Fancy</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Selection</th>
                  <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Runs/Odds</th>
                  <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Payout</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Event ID</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Match ID</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {rows.map((r) => (
                  <tr key={r.id} className="hover:bg-gray-50 transition-colors">
                    <td className="px-4 py-3 whitespace-nowrap">{r.userid}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.fancyname}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.selection}</td>
                    <td className="px-4 py-3 text-right whitespace-nowrap">{r.runsodds}</td>
                    <td className="px-4 py-3 text-right whitespace-nowrap">{formatCurrency(r.payout)}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.eventid}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.matchid}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{formatDate(r.created_at)}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {/* Pagination */}
        {pagination.totalPages > 1 && (
          <div className="flex items-center justify-center mt-6">
            <button
              onClick={() => handlePageChange(pagination.currentPage - 1)}
              disabled={pagination.currentPage === 1}
              className="px-4 py-2 mr-4 bg-gray-100 rounded border hover:bg-gray-200 disabled:opacity-50"
            >
              Previous
            </button>
            <div className="text-center px-4 py-2 bg-gray-50 rounded border min-w-[80px]">
              {pagination.currentPage} / {pagination.totalPages || 1}
            </div>
            <button
              onClick={() => handlePageChange(pagination.currentPage + 1)}
              disabled={pagination.totalPages === 0 || pagination.currentPage === pagination.totalPages}
              className="px-4 py-2 ml-4 bg-gray-100 rounded border hover:bg-gray-200 disabled:opacity-50"
            >
              Next
            </button>
          </div>
        )}
      </div>
    </div>
  );
};

export default FanWins;
