import React, { useEffect, useMemo, useState } from "react";
import axios from "axios";

interface MoMatch {
  matchId: string;
  eventId: string;
  matchTitle: string;
  teamOne: string;
  teamTwo: string;
  totalBets: number;
  counts: number;
}

interface MoMatchesResponse {
  success: boolean;
  data: MoMatch[];
}

interface CalculatePayoutsResponse {
  success: boolean;
  totalusers: number;
  total_credit: number;
  total_lability: number;
  details?: Array<{
    user_id: number;
    finalcredit: number;
    mostNeg: number;
    normalizedMostNeg: number;
    exposures: Record<string, number>;
  }>;
}

type WinnerKey = "team1" | "team2" | "draw" | "refund";

const MarketInternalSettle: React.FC = () => {
  const staffId = localStorage.getItem("currentUserId") || "";
  const token = localStorage.getItem("token") || "";

  const [rows, setRows] = useState<MoMatch[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState("");

  const [search, setSearch] = useState("");
  const [selected, setSelected] = useState<MoMatch | null>(null);
  const [winnerKey, setWinnerKey] = useState<WinnerKey | null>(null);

  const [calcLoading, setCalcLoading] = useState(false);
  const [calcError, setCalcError] = useState("");
  const [calcData, setCalcData] = useState<CalculatePayoutsResponse | null>(
    null
  );

  const [submitLoading, setSubmitLoading] = useState(false);
  const [submitError, setSubmitError] = useState<string | null>(null);
  const [submitSuccess, setSubmitSuccess] = useState<string | null>(null);

  const BASE_URL = useMemo(() => {
    return (
      (import.meta as any)?.env?.VITE_API_BASE_URL ||
      (process.env as any)?.REACT_APP_API_BASE_URL ||
      "https://api.bitcoinjito.com"
    );
  }, []);

  const headers: Record<string, string> = {
    "Content-Type": "application/json",
    ...(token ? { Authorization: `Bearer ${token}` } : {}),
    ...(staffId ? { "x-staff-id": staffId } : {}),
  };

  const fetchMatches = async () => {
    setLoading(true);
    setError("");
    try {
      const { data } = await axios.get<MoMatchesResponse>(
        `${BASE_URL}/sportsbetting/momatches`,
        { headers }
      );
      setRows(data?.data || []);
    } catch (err: any) {
      console.error("fetchMatches error:", err);
      setError(err?.response?.data?.message || "Failed to load matches");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchMatches();
  }, [BASE_URL]);

  const filteredRows = rows.filter((r) => {
    const q = search.trim().toLowerCase();
    if (!q) return true;
    return (
      r.matchTitle.toLowerCase().includes(q) ||
      r.eventId.toLowerCase().includes(q) ||
      r.teamOne.toLowerCase().includes(q) ||
      r.teamTwo.toLowerCase().includes(q) ||
      r.matchId.toLowerCase().includes(q)
    );
  });

  const openModal = async (row: MoMatch, w: WinnerKey) => {
    setSelected(row);
    setWinnerKey(w);
    setCalcData(null);
    setCalcError("");
    setCalcLoading(true);
    setSubmitError(null);
    setSubmitSuccess(null);

    const winnerName =
      w === "team1"
        ? row.teamOne
        : w === "team2"
        ? row.teamTwo
        : w === "draw"
        ? "The Draw"
        : "refund";

    try {
      const { data } = await axios.post<CalculatePayoutsResponse>(
        `${BASE_URL}/sportsbetting/calculate-payouts`,
        {
          matchId: row.matchId,
          eventId: row.eventId,
          winnerName,
          teamOne: row.teamOne,
          teamTwo: row.teamTwo,
          counts: row.counts,
        },
        { headers }
      );
      setCalcData(data);
    } catch (err: any) {
      console.error("calculate-payouts error:", err);
      setCalcError(err?.response?.data?.message || "Failed to calculate payouts");
    } finally {
      setCalcLoading(false);
    }
  };

  const closeModal = () => {
    setSelected(null);
    setWinnerKey(null);
    setCalcData(null);
    setCalcError("");
    setSubmitError(null);
    setSubmitSuccess(null);
    setCalcLoading(false);
    setSubmitLoading(false);
  };

  const handleSubmit = async () => {
    setSubmitError(null);
    setSubmitSuccess(null);

    if (!selected || !winnerKey) {
      setSubmitError("No selection is active.");
      return;
    }

    const winnerName =
      winnerKey === "team1"
        ? selected.teamOne
        : winnerKey === "team2"
        ? selected.teamTwo
        : winnerKey === "draw"
        ? "The Draw"
        : "refund";

    const body = {
      matchId: selected.matchId,
      eventId: selected.eventId,
      winnerName,
      teamOne: selected.teamOne,
      teamTwo: selected.teamTwo,
      counts: selected.counts,
    };

    setSubmitLoading(true);
    try {
      const { data } = await axios.post(
        `${BASE_URL}/sportsbetting/manual-settlement`,
        body,
        { headers }
      );

      if (data && (data.success === true || data.success === "true")) {
        setSubmitSuccess(data.message || "Market settled successfully.");
        await fetchMatches();
        closeModal();
      } else {
        setSubmitError(
          data?.message || "Manual settlement returned an unexpected response."
        );
      }
    } catch (err: any) {
      console.error("manual-settlement error:", err);
      setSubmitError(
        err?.response?.data?.message || "Failed to perform manual settlement"
      );
    } finally {
      setSubmitLoading(false);
    }
  };

  const formatNumber = (n: number | null | undefined) =>
    typeof n === "number" && !Number.isNaN(n)
      ? new Intl.NumberFormat().format(n)
      : "-";

  return (
    <div className="ml-0 lg:ml-[256px] p-6">
      <div className="bg-white rounded-lg shadow-md p-6">
        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4 mb-6">
          <h1 className="text-2xl font-bold">Market Internal Settle</h1>
          <div className="w-full md:w-80">
            <input
              type="text"
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              placeholder="Search match, team, event, or match id"
              className="w-full p-2 border rounded focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
        </div>

        {loading && (
          <div className="flex items-center justify-center h-40 text-gray-500">
            Loading matches...
          </div>
        )}

        {error && !loading && (
          <div className="mb-4 p-3 rounded bg-red-50 border border-red-200 text-red-700">
            {error}
          </div>
        )}

        {!loading && !error && filteredRows.length === 0 && (
          <div className="text-center text-gray-500 py-10">No matches found.</div>
        )}

        {!loading && !error && filteredRows.length > 0 && (
          <div className="overflow-x-auto">
            <table className="w-full bg-white shadow-sm rounded-lg overflow-hidden">
              <thead className="bg-gray-100 border-b">
                <tr>
                  <th className="px-4 py-3 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                    Match
                  </th>
                  <th className="px-4 py-3 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                    Teams
                  </th>
                  <th className="px-4 py-3 text-right text-xs font-semibold text-gray-600 uppercase tracking-wider">
                    Total Bets
                  </th>
                  <th className="px-4 py-3 text-center text-xs font-semibold text-gray-600 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {filteredRows.map((r) => (
                  <tr
                    key={`${r.matchId}-${r.eventId}`}
                    className="hover:bg-gray-50"
                  >
                    <td className="px-4 py-3">
                      <div className="font-medium text-gray-900">
                        {r.matchTitle}
                      </div>
                      <div className="text-xs text-gray-500">
                        Match ID: {r.matchId} • Event ID: {r.eventId}
                      </div>
                    </td>
                    <td className="px-4 py-3">
                      <div className="text-gray-900">
                        {r.teamOne}{" "}
                        <span className="text-gray-400">vs</span>{" "}
                        {r.teamTwo}
                      </div>
                    </td>
                    <td className="px-4 py-3 text-right">
                      {formatNumber(r.totalBets)}
                    </td>
                    <td className="px-4 py-3">
                      {r.counts === 2 || r.counts === 3 ? (
                        <div className="flex items-center justify-center gap-2">
                          <button
                            onClick={() => openModal(r, "team1")}
                            className="px-3 py-1.5 rounded bg-blue-600 text-white text-sm hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-400"
                          >
                            Team 1
                          </button>

                          <button
                            onClick={() => openModal(r, "team2")}
                            className="px-3 py-1.5 rounded bg-emerald-600 text-white text-sm hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-emerald-400"
                          >
                            Team 2
                          </button>

                          {r.counts === 3 && (
                            <button
                              onClick={() => openModal(r, "draw")}
                              className="px-3 py-1.5 rounded bg-gray-700 text-white text-sm hover:bg-gray-800 focus:outline-none focus:ring-2 focus:ring-gray-400"
                            >
                              Draw
                            </button>
                          )}

                          {/* 👇 NEW VOID BUTTON */}
                          <button
                            onClick={() => openModal(r, "refund")}
                            className="px-3 py-1.5 rounded bg-red-600 text-white text-sm hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-red-400"
                          >
                            Void
                          </button>
                        </div>
                      ) : (
                        <div className="text-center text-xs text-gray-500">
                          No actions (counts ≠ 2 or 3)
                        </div>
                      )}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
            <div className="text-sm text-gray-500 mt-3">
              Showing {filteredRows.length} of {rows.length} matches
            </div>
          </div>
        )}
      </div>

      {selected && winnerKey && (
        <div
          className="fixed inset-0 z-50 flex items-center justify-center p-4"
          role="dialog"
          aria-modal="true"
        >
          <div
            className="absolute inset-0 bg-black bg-opacity-40"
            onClick={closeModal}
          />

          <div className="relative bg-white w-full max-w-xl rounded-2xl shadow-xl p-6">
            <div className="flex items-start justify-between mb-4">
              <div>
                <h2 className="text-xl font-semibold">Manual Settlement</h2>
                <p className="text-sm text-gray-500">{selected.matchTitle}</p>
                <p className="text-xs text-gray-500">
                  {selected.teamOne} <span className="text-gray-400">vs</span>{" "}
                  {selected.teamTwo}
                </p>
              </div>
              <button
                onClick={closeModal}
                className="ml-4 rounded-full p-2 hover:bg-gray-100 focus:outline-none"
                aria-label="Close"
              >
                ✕
              </button>
            </div>

            <div className="mb-4">
              <div className="text-sm">
                <span className="text-gray-500 mr-1">Chosen Winner:</span>
                <span className="font-medium">
                  {winnerKey === "team1"
                    ? selected.teamOne
                    : winnerKey === "team2"
                    ? selected.teamTwo
                    : winnerKey === "draw"
                    ? "The Draw"
                    : "Refund (Void)"}
                </span>
              </div>
              <div className="text-xs text-gray-500">
                Match ID: {selected.matchId} • Event ID: {selected.eventId}
              </div>
            </div>

            <div className="min-h-[120px]">
              {calcLoading && (
                <div className="flex items-center justify-center h-28 text-gray-500">
                  Calculating...
                </div>
              )}

              {calcError && !calcLoading && (
                <div className="p-3 rounded bg-red-50 border border-red-200 text-red-700 mb-3">
                  {calcError}
                </div>
              )}

              {calcData && calcData.success && (
                <div className="grid grid-cols-1 sm:grid-cols-3 gap-3">
                  <div className="border rounded-lg p-3 bg-gray-50">
                    <div className="text-xs text-gray-500">Total Users</div>
                    <div className="text-lg font-semibold">
                      {formatNumber(calcData.totalusers)}
                    </div>
                  </div>
                  <div className="border rounded-lg p-3 bg-gray-50">
                    <div className="text-xs text-gray-500">Total Credit</div>
                    <div className="text-lg font-semibold">
                      {formatNumber(calcData.total_credit)}
                    </div>
                  </div>
                  <div className="border rounded-lg p-3 bg-gray-50">
                    <div className="text-xs text-gray-500">
                      Total Liability
                    </div>
                    <div className="text-lg font-semibold">
                      {formatNumber(calcData.total_lability)}
                    </div>
                  </div>
                </div>
              )}
            </div>

            {submitError && (
              <div className="mt-4 p-3 rounded bg-red-50 border border-red-200 text-red-700">
                {submitError}
              </div>
            )}
            {submitSuccess && (
              <div className="mt-4 p-3 rounded bg-green-50 border border-green-200 text-green-700">
                {submitSuccess}
              </div>
            )}

            <div className="flex justify-end gap-2 mt-4">
              <button
                onClick={closeModal}
                className="px-4 py-2 bg-gray-200 rounded hover:bg-gray-300"
                disabled={submitLoading}
              >
                Cancel
              </button>
              <button
                onClick={handleSubmit}
                disabled={submitLoading}
                className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 disabled:opacity-50"
              >
                {submitLoading ? "Submitting..." : "Submit"}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default MarketInternalSettle;
