import React, { useEffect, useState } from "react";
import { apiFetch } from "../utils/api";
import { ROLE_HIERARCHY } from "../components/admin-management/UserManagementTab";
import { Users } from "lucide-react";

interface Node {
  id: number;
  name: string;
  role: keyof typeof ROLE_HIERARCHY;
  percentage: number;
  parent_id: number | null;
  depth: number;
}

export default function PercentageHierarchyPage() {
  const [tree, setTree] = useState<Node[]>([]);
  const [loading,setLoading] = useState(true);

 useEffect(() => {
  (async () => {
    try {
      const me = Number(localStorage.getItem("currentUserId"));

      /* ↓↓↓ 1. full subtree (down‑line) */
      const subtree = await apiFetch<Node[]>(`/api/staff/${me}/percent-tree`);

      /* ↓↓↓ 2. upstream chain (SuperAdmin ➜ … ➜ me)            */
      const chain   = await apiFetch<Node[]>(`/api/staff/${me}/percent-chain`);

      /* keep only *my direct parent* (depth = 1) */
      const parent  = chain.find(n => n.depth === 1);

      /* final tree = parent (if any) ➜ me ➜ kids ➜ grand‑kids … */
      const combined: Node[] = parent
        ? [ { ...parent, depth: -1 }, ...subtree ] // put parent one level above
        : subtree;

      setTree(combined);
    } finally {
      setLoading(false);
    }
  })();
}, []);


  if (loading) return <div className="text-center py-20 text-gray-400">Loading…</div>;

  /* helpers */
  const idSet   = new Set(tree.map(n => n.id));
  const byParent: Record<number | "root", Node[]> = { root:[] };
  tree.forEach(n => {
    const key = n?.parent_id == null || !idSet.has(n?.parent_id) ? "root" : n?.parent_id;
    byParent[key] = byParent[key] || [];
byParent[key].push(n);
  });

  const FlowNode: React.FC<{ node: Node; isRoot?: boolean }> = ({ node, isRoot }) => {
    const kids = byParent[node.id] || [];
    const [open,setOpen] = useState(true);
    const roleCfg = ROLE_HIERARCHY[node.role];
    return (
      <div className="flex flex-col items-center">
        {/* node card */}
        <div
          className={`${roleCfg.color} text-white px-3 py-2 rounded-lg
                      shadow-md min-w-[160px] text-center cursor-pointer
                      hover:shadow-lg transition-all`}
          onClick={() => kids.length && setOpen(!open)}
        >
          <div className="text-xs font-semibold truncate">{node.name}</div>
          <div className="text-[10px] opacity-90">{node.role}</div>
          {isRoot && <div className="text-[10px] mt-0.5">(root)</div>}
        </div>

        {/* connector with % */}
        {kids.length && open && (
          <div className="flex flex-col items-center">
            <div className="w-0.5 h-4 bg-gray-300"></div>
          </div>
        )}

        {/* children */}
        {kids.length && open && (
          <div className="relative">
            {kids.length > 1 && (
              <div className="absolute top-0 left-1/2 -translate-x-1/2
                              w-full h-0.5 bg-gray-300"></div>
            )}
            <div className="flex flex-wrap justify-center gap-6 pt-2">
              {kids.map(c => (
                <div key={c.id} className="relative">
                  <div className="absolute -top-3 left-1/2 -translate-x-1/2
                                  bg-white text-[10px] px-1 rounded shadow">
                    {c.percentage} %
                  </div>
                  <FlowNode node={c}/>
                </div>
              ))}
            </div>
          </div>
        )}
      </div>
    );
  };

  return (
    <div className='ml-0 lg:ml-[256px]'>
    <div className="p-6 max-w-7xl mx-auto">
      <header className="flex items-center gap-3 mb-8">
        <Users className="w-6 h-6 text-blue-600"/>
        <h1 className="text-2xl font-semibold text-white">Percentage Hierarchy</h1>
      </header>

      <div className="bg-gradient-to-br from-blue-50 to-indigo-50
                      border border-blue-100 rounded-xl p-6 overflow-auto">
        <div className="flex justify-center">
          {(byParent.root || []).map(r => (
            <FlowNode key={r.id} node={r} isRoot />
          ))}
        </div>
      </div>
    </div>
    </div>
  );
}
