
import React, { useEffect, useRef, useState } from "react";
import { NavLink, useNavigate, useLocation } from "react-router-dom";
import {
  LayoutDashboard,
  Users,
  CreditCard,
  Wallet,
  History,
  LogOut,
  ArrowUpCircle,
  ArrowDownCircle,
  Percent,
  Bell,
  Gamepad,
  Link as LinkIcon,
  Gift,
  Settings,
  ChevronsRight,
  Shield,
  File,
  Gamepad2,
  Settings2Icon,
  ArrowBigUpIcon
} from "lucide-react";
// import logo from "../assets/logo.png";

/* ------------------------------------------------------------------
   Sidebar role gating
   ------------------------------------------------------------------ */
const GROUP_ACCESS: Record<string, string[]> = {
  SuperAdmin: ["Admin","Main","History", "Gaming", "Marketing","Transactions","Settings","Assets Management","Services",'Self Settlement'],
  Admin     : ["Admin", "History", "Gaming", "Marketing","Settings"],
  SubAdmin  : ["Admin", "History", "Gaming", "Marketing"],
  Master    : ["Admin", "History", "Gaming", "Marketing"],
  Agent     : ["Admin", "History", "Gaming", "Marketing"],
  SubAgent  : ["Admin", "History", "Gaming", "Marketing"]
};

interface SidebarProps {
  isOpen: boolean;
  toggleSidebar: () => void;
}

interface MenuItem {
  path: string;
  label: string;
  icon: React.ReactNode;
  requireRole?: string;
}
interface MenuGroup {
  title: string;
  items: MenuItem[];
}

/* ------------------------------------------------------------------
   Component
   ------------------------------------------------------------------ */
const Sidebar: React.FC<SidebarProps> = ({ isOpen, toggleSidebar }) => {
  const navigate      = useNavigate();
  const location      = useLocation();
  const [collapsed, setCollapsed] = useState<Record<string, boolean>>({});
  const sidebarRef    = useRef<HTMLDivElement>(null);

  const userRole      = localStorage.getItem("userRole") || "Guest";
  const allowedGroups = GROUP_ACCESS[userRole as keyof typeof GROUP_ACCESS] ?? [];

  /* -------- menu definition -------- */
  const menuGroups: MenuGroup[] = [
    {
      title: "Admin",
      items: [
        { path: "/admin-management", label: "Client Management", icon: <Shield size={18} /> },
          { path: "/percentage-hierarchy", label: "Percentage Hierarchy", icon: <ArrowBigUpIcon size={18} /> }
      ]
    },
    {
      title: "Main",
      items: [
        { path: "/admin-dashboard", label: "Dashboard",   icon: <LayoutDashboard size={18} /> },
        { path: "/users",            label: "Users",       icon: <Users size={18} /> },
        { path: "/reports",          label: "User Reports",icon: <File  size={18} /> },
        { path: "/kyc",              label: "User KYC",    icon: <File  size={18} /> },
          { path: "/wallet",      label: "Wallet",         icon: <Wallet size={18} /> },
           { path: "/swaphistory", label: "Swap History",   icon: <CreditCard size={18} /> },
                   { path: "/affiliate",   label: "Affiliate",      icon: <LinkIcon size={18} /> },
        { path: "/redeemcode",  label: "Redeem Codes",   icon: <Gift     size={18} /> },
            { path: "/exchangerate", label: "Exchange Rate",     icon: <CreditCard size={18} /> }
      ]
    },
    {
      title: "History",
      items: [
        { path: "/deposit",     label: "Deposit History",        icon: <ArrowUpCircle  size={18} /> },
        { path: "/withdraw",    label: "Withdraw History",       icon: <ArrowDownCircle size={18} /> },
         { path: "/marketwins",    label: "Market History",       icon: <ArrowDownCircle size={18} /> },
          { path: "/fanwins",    label: "Fancy History",       icon: <ArrowDownCircle size={18} /> },
      
        { path: "/history",     label: "Bet History",    icon: <History size={18} /> },
        { path: "/sportsbeting",     label: "Sports Bet",    icon: <History size={18} /> },
        { path: "/fancyreport",     label: "Fancy Api Report",    icon: <History size={18} /> },

      ]
    },
    {
      title: "Self Settlement",
      items: [
        { path: "/MOsettle",     label: "market settlement",        icon: <ArrowUpCircle  size={18} /> },
         { path: "/Fansettle",     label: "Fancy settlement",        icon: <ArrowUpCircle  size={18} /> },
      ]
    },
     {
      title: "Transactions",
      items: [
        { path: "/depositself",     label: "Deposit",        icon: <ArrowUpCircle  size={18} /> },
        { path: "/withdrawself",    label: "Withdraw",       icon: <ArrowDownCircle size={18} /> },
      ]
    },
    // {
    //   title: "Gaming",
    //   items: [
    //     { path: "/house",        label: "House Edge",    icon: <Percent size={18} /> },
    //     { path: "/sportsbeting", label: "Sports Bets",   icon: <Gamepad size={18} /> },
    //     { path: "/club",         label: "Club",          icon: <Users   size={18} /> }
    //   ]
    // },
    // {
    //   title: "Marketing",
    //   items: [
    //     { path: "/affiliate",   label: "Affiliate",      icon: <LinkIcon size={18} /> },
    //     { path: "/redeemcode",  label: "Redeem Codes",   icon: <Gift     size={18} /> },

    //   ]
    // },
     {
      title: "Settings",
      items: [
        // { path: "/siteconfig",   label: "Site Config",       icon: <Settings  size={18} /> },
        { path: "/bankdetails",  label: "Bank Details",      icon: <CreditCard size={18} /> },
          { path: "/userconfig",  label: "User Site Config",      icon: <Settings2Icon size={18} /> },
        // { path: "/exchangerate", label: "Exchange Rate",     icon: <CreditCard size={18} /> }
      ]
    },
    {
      title: "Assets Management",
      items: [
        { path: "/banner",       label: "Asset Management",  icon: <CreditCard size={18} /> }
      ]},
       {
      title: "Services",
      items: [
        // { path: "/winloss",       label: "Original game management",  icon: <Gamepad2 size={18} /> },
        { path: "/siteconfig",       label: "Global Site Config",  icon: <Settings2Icon size={18} /> },
        { path: "/notification",label: "Notification",   icon: <Bell     size={18} /> }
      ]}
  ];

  /* -------- apply role filter -------- */
  const filteredMenuGroups = menuGroups
    .filter(g => allowedGroups.includes(g.title))
    .map(g => ({
      ...g,
      items: g.items.filter(i => !i.requireRole || i.requireRole === userRole)
    }));

  /* -------- auto-expand active group -------- */
  useEffect(() => {
    filteredMenuGroups.forEach(group => {
      const activeInside = group.items.some(item => item.path === location.pathname);
      if (activeInside) {
        setCollapsed(prev => ({ ...prev, [group.title]: false }));
      }
    });
  }, [location.pathname]);

  /* -------- collapse on outside click (mobile) -------- */
  useEffect(() => {
    const handleClickOutside = (evt: MouseEvent) => {
      if (isOpen && sidebarRef.current && !sidebarRef.current.contains(evt.target as Node)) {
        toggleSidebar();
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, [isOpen]);

  /* -------- helpers -------- */
  const toggleGroup = (title: string) =>
    setCollapsed(prev => ({ ...prev, [title]: !prev[title] }));

  const handleLogout = () => {
    const admins = localStorage.getItem("adminUsers"); // preserve demo users
    localStorage.clear();
    if (admins) localStorage.setItem("adminUsers", admins);
    navigate("/login");
    toggleSidebar();
  };

  /* -------- render -------- */
  return (
    <div
      ref={sidebarRef}
      className={`w-64 h-screen bg-[#1A2033] shadow-lg fixed top-0 left-0 z-50 transition-transform duration-300 ${
        isOpen ? "translate-x-0" : "-translate-x-full"
      } lg:translate-x-0 flex flex-col border-r border-[#2D334A]`}
    >
      {/* Top bar logo */}
      <div className="p-4 flex items-center justify-between border-b border-[#2D334A]">
        {/* <img src={logo} alt="Logo" className="h-8 w-auto" /> */}
        <button onClick={toggleSidebar} className="text-gray-400 hover:text-white lg:hidden">
          <LogOut size={20} />
        </button>
      </div>

      {/* User badge */}
      <div className="p-4 border-b border-[#2D334A]">
        <div className="flex items-center space-x-3">
          <div className="w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center text-white font-medium">
            {localStorage.getItem("userName")?.charAt(0) || "A"}
          </div>
          <div>
            <p className="text-white text-sm font-medium">
              {localStorage.getItem("userName") || "Admin"}
            </p>
            <p className="text-gray-400 text-xs">{userRole}</p>
          </div>
        </div>
      </div>

      {/* Groups */}
      <div className="flex-1 overflow-y-auto py-2 px-2">
        {filteredMenuGroups.map(group => (
          <div key={group.title} className="mb-3">
            <button
              onClick={() => toggleGroup(group.title)}
              className="w-full flex items-center justify-between px-3 py-2 text-xs font-medium text-gray-400 uppercase tracking-wider"
            >
              <span>{group.title}</span>
              <ChevronsRight
                size={14}
                className={`transition-transform ${!collapsed[group.title] ? "rotate-90" : ""}`}
              />
            </button>

            <div
              className={`space-y-1 mt-1 overflow-hidden transition-all ${
                collapsed[group.title] ? "max-h-0 opacity-0" : "max-h-96 opacity-100"
              }`}
            >
              {group.items.map(item => (
                <NavLink
                  key={item.path}
                  to={item.path}
                  onClick={window.innerWidth < 1024 ? toggleSidebar : undefined}
                  className={({ isActive }) =>
                    `flex items-center px-3 py-2 rounded-md text-sm transition ${
                      isActive
                        ? "bg-[#646ECD]/20 text-[#646ECD]"
                        : "text-gray-300 hover:bg-[#2B3141] hover:text-white"
                    }`
                  }
                >
                  <span className="mr-3">{item.icon}</span>
                  <span>{item.label}</span>
                </NavLink>
              ))}
            </div>
          </div>
        ))}
      </div>

      {/* Logout */}
      <div className="mt-auto p-3 border-t border-[#2D334A]">
        <button
          onClick={handleLogout}
          className="flex items-center w-full px-3 py-2 text-sm text-gray-300 hover:bg-[#2B3141] hover:text-white rounded-md"
        >
          <LogOut size={18} className="mr-3" />
          Log Out
        </button>
      </div>
    </div>
  );
};

export default Sidebar;
